<?php

namespace Incevio\Package\Wallet\Http\Controllers;

use App\Models\Customer;
use App\Models\Shop;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Http\RedirectResponse;
use Incevio\Package\Wallet\Models\Transaction;
use Incevio\Package\Wallet\Traits\HasTransaction;
use Incevio\Package\Wallet\Services\CommonService;
use Incevio\Package\Wallet\Http\Requests\TransferRequest;

class TransferController extends Controller
{
    use HasTransaction;

    /**
     * @param Request $request
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function show_form(Request $request)
    {
        $wallet = self::getWallet();

        if (Auth::guard('customer')->check()) {
            $tab = 'wallet';

            $content = view('wallet::customer._transfer', compact('wallet'))->render();

            return view('theme::dashboard', compact('tab', 'content'));
        }

        return view('wallet::_transfer', compact('wallet'));
    }

    /**
     * @param Request $request
     * @return \Illuminate\Contracts\Foundation\Application|\Illuminate\Contracts\View\Factory|\Illuminate\Contracts\View\View
     * for transfer to own customer or merchant account
     */
    public function show_self_form(Request $request)
    {
        $wallet = self::getWallet();

        if (Auth::guard('customer')->check()) {
            $tab = 'wallet';

            $content = view('wallet::customer._self_transfer', compact('wallet'))->render();

            return view('theme::dashboard', compact('tab', 'content'));
        }

        return view('wallet::_self_transfer', compact('wallet'));
    }

    /**
     * Transfer balance to Other Customer
     * @param TransferRequest $request
     * @return RedirectResponse
     */
    public function transfer(TransferRequest $request)
    {
        $user_type = Auth::guard('customer')->check() ? 'customer' : 'shop';

        if (!customer_can_register() && !$request->has('email')) {
            if ($user_type == 'customer') {
                $email = Auth::guard('customer')->user()->email;
                $toWallet = Shop::where('email', $email)->first();
            } elseif ($user_type == 'shop') {
                $email = Auth::guard('web')->user()->shop->email;
                $toWallet = Customer::where('email', $email)->first();
            }
        } else {
            $toWallet = self::getWallet($user_type, $request->email);
        }

        if (!$toWallet) {
            return redirect()->back()->withInput()
                ->with('warning', trans('wallet::lang.email_not_found'));
        }

        try {
            $fromWallet = self::getWallet();

            $meta = self::getMeta($request, $toWallet, $fromWallet);

            (new CommonService())->transfer($fromWallet->wallet, $toWallet->wallet, $request->amount, $meta);
        } catch (\Exception $exception) {
            Log::error('Transfer failed:: ');
            Log::info($exception);

            return redirect()->route(self::getRouteName())
                ->with('warning', $exception->getMessage())->withInput();
        }

        return redirect()->route(self::getRouteName('wallet'))
            ->with('success', trans('wallet::lang.transfer_success'));
    }

    /**
     * get Formatted meta:
     * @param $request
     * @param $customer
     * @return array[]
     */
    private function getMeta($request, $to, $from)
    {
        return [
            'from' => [
                'type' => Transaction::TYPE_WITHDRAW,
                'to' => $to->email,
                'description' => trans('wallet::lang.balance_sent_to', ['email' => $request->email])
            ],
            'to' => [
                'type' => Transaction::TYPE_DEPOSIT,
                'from' => $from->email,
                'description' => trans('wallet::lang.balance_received_from', ['email' => $from->email])
            ]
        ];
    }
}
